
from libcloud.compute.drivers import AzureNodeDriver

CUSTOM_RESOURCE_API_VERSION = '2020-06-01'

class CustomAzureNodeDriver(AzureNodeDriver):
    """Custom (Royal Render) node driver for Azure Resource Manager."""

    def __init__(self, tenant_id, subscription_id, key, secret,
                 secure=True, host=None, port=None,
                 api_version=None, region=None, **kwargs):

        super(CustomAzureNodeDriver, self).__init__(key=key, secret=secret,
                                              secure=secure,
                                              host=host, port=port,
                                              api_version=api_version,
                                              region=region, **kwargs)
    
    def ex_list_skus(self):
        pass

    def ex_create_resourcegroup(self, ex_resource_group, location=None):
        if location is None:
            if self.default_location:
                location = self.default_location
            else:
                raise ValueError("location is required.")

        target = \
            "/subscriptions/%s/resourcegroups/%s" \
            % (self.subscription_id, ex_resource_group)

        data = {
          "location": location.id
        }

        r = self.connection.request(
            target,
            params={"api-version": CUSTOM_RESOURCE_API_VERSION},
            data=data,
            method="PUT")

    def ex_delete_resourcegroup(self, ex_resource_group):
        target = \
            "/subscriptions/%s/resourcegroups/%s" \
            % (self.subscription_id, ex_resource_group)

        r = self.connection.request(
            target,
            params={"api-version": CUSTOM_RESOURCE_API_VERSION},
            method="DELETE")

    def ex_get_network(self, name, ex_resource_group):
        """
        Return a Network object based on a network name.
        :param  name: The name or URL of the network
        :type   name: ``str``
        :return:  A Network object for the network
        :rtype:   :class:`GCENetwork`
        """
        if name.startswith('https://'):
            target = name
        else:
            target = \
            "/subscriptions/%s/resourcegroups/%s" \
            "/providers/Microsoft.Network/virtualNetworks/%s" \
            % (self.subscription_id, ex_resource_group, name)

        response = self.connection.request(request, method='GET').object
        return self._to_network(response)

    def ex_create_network(self, name, cidr, ex_resource_group, location=None):
        if location is None:
            if self.default_location:
                location = self.default_location
            else:
                raise ValueError("location is required.")

        target = \
            "/subscriptions/%s/resourcegroups/%s" \
            "/providers/Microsoft.Network/virtualNetworks/%s" \
            % (self.subscription_id, ex_resource_group, name)

        data = {
          "properties": {
            "addressSpace": {
              "addressPrefixes": [
                cidr
              ]
            }
          },
          "location": location.id
        }

        r = self.connection.request(
            target,
            params={"api-version": CUSTOM_RESOURCE_API_VERSION},
            data=data,
            method="PUT")

        return self.ex_get_network(name, ex_resource_group)

    
    def ex_get_subnetwork(self, name, ex_resource_group, network):
        """
        Return a Subnetwork object based on name and region.
        :param  name: The name or URL of the subnetwork
        :type   name: ``str``
        :keyword ex_resource_group: The resource group of the subnetwork
        :type   ex_resource_group: ``str``
        :return:  A Subnetwork object
        :rtype:   :class:`GCESubnetwork`
        """
        if not network:
            raise ValueError("Must provide a network for the subnetwork.")
        else:
            if isinstance(network, GCENetwork):
                network = self.ex_get_network(network)

        if name.startswith('https://'):
            target = name
        else:
            target = \
                "/subscriptions/%s/resourcegroups/%s" \
                "/providers/Microsoft.Network/virtualNetworks/%s" \
                "/subnets/%s" \
                % (self.subscription_id, ex_resource_group, network.name, name)

        response = self.connection.request(request, method='GET').object
        return self._to_subnetwork(response)

    def ex_create_subnetwork(self, name, cidr, network):
        if not cidr:
            raise ValueError("Must provide an IP network in CIDR notation.")

        if not network:
            raise ValueError("Must provide a network for the subnetwork.")
        else:
            if isinstance(network, GCENetwork):
                network = self.ex_get_network(network)

        target = \
            "/subscriptions/%s/resourcegroups/%s" \
            "/providers/Microsoft.Network/virtualNetworks/%s" \
            "/subnets/%s" \
            % (self.subscription_id, ex_resource_group, network.name, name)

        data = {
          "properties": {
            "addressPrefix": cidr
          }
        }

        r = self.connection.request(
            target,
            params={"api-version": CUSTOM_RESOURCE_API_VERSION},
            data=data,
            method="PUT")

        return self.ex_get_subnetwork(name, ex_resource_group, network)

    def ex_create_firewall_rule(self):
        pass

    def ex_create_route(self):
        pass


    def _to_network(self, network):
        """
        Return a Network object from the JSON-response dictionary.
        :param  network: The dictionary describing the network.
        :type   network: ``dict``
        :return: Network object
        :rtype: :class:`GCENetwork`
        """
        extra = {}

        extra['id'] = network.get('id')

        return GCENetwork(id=network['id'], name=network['name'],
                          cidr=network.get('properties').get('addressSpace').get('addressPrefixes')[0], driver=self,
                          extra=extra)


    
    def _to_subnetwork(self, subnetwork, network):
        """
        Return a Subnetwork object from the JSON-response dictionary.
        :param  subnetwork: The dictionary describing the subnetwork.
        :type   subnetwork: ``dict``
        :return: Subnetwork object
        :rtype: :class:`GCESubnetwork`
        """
        extra = {}

        extra['selfLink'] = subnetwork.get('id')

        return GCESubnetwork(id=subnetwork['id'], name=subnetwork['name'],
                             cidr=subnetwork.get('properties').get('addressPrefix'),
                             network=network, driver=self,
                             extra=extra)