#
# This script shows the usage of the libpyRR (rrLib) module.
# - It connects to the rrServer
# - Displays all clients with their CPU usage
# - Displays all jobs (or jobs of user X)
# - Displays all users set in rrConfig/rrLogins
#
#

import sys
import os

mod_dir = os.path.dirname(__file__)
if mod_dir not in sys.path:
    sys.path.append(mod_dir)

from rr_python_utils.load_rrlib import rrLib  # module libpyRR3 for python 3, libpyRR2 for python 2
import rr_python_utils.connection as rr_connect
import rrJob

print("rrLib version: "+rrLib.getRRModuleVersion())




if sys.version_info.major < 3:
    RANGE_FUNC = xrange
else:
    RANGE_FUNC = range


class InfoLevel:
    """There are three levels of job information:
        MinInfo: The list, all jobs with ID, user, software, project
        Basic:   Scene information
        Info:    All job data you can have
        
        MinInfo is received for ALL jobs that do not fit the filter you set.
        Basic is useful if you want to save RAM or if your network connection is slow
    """
    MinInfo = 0
    Basic = 1
    Info = 2


def print_clients(tcp):
    print("\nCheck clients")
    if not tcp.clientGetList():
        print("Error getting clients: " + tcp.errorMessage())
        return

    clients = tcp.clients
    clients_count = clients.count()
    print("Number of client found: " + str(clients_count))

    for i in RANGE_FUNC(clients_count):
        client = clients.at(i)
        print("\tCpuUsage {0:6.2f} name: {1}".format(client.status.CPU_Usage, client.status.name))


def print_jobs(tcp, job_user_name="", info_level=InfoLevel.Basic):
    ### --------------------------------------------------------------------- JOBS
    ### IMPORTANT: tcp caches all jobs, if you destroy it, the rrServer has to send all jobs again
    ### A list request will always be executed, so you have a list of all jobs at the rrServer
    ### But to reduce traffic, you should set a filter for the jobs you need before you request more data

    ### How does it work?
    ### Think about it the same way rrControl works
    ### a) In rrControl you set the filter (Project, user, finished, rendering)
    ### b) Then rrControl/you pressed the update button in the upper right to retrieve the jobs
    ### c) rrControl has now your list of jobs matching the filter.
    ###    and rrControl has a list of all jobs in the queue. (You notice that if you change the filter, but rrControl cannot show job data like scene name)
    
    print("\nCheck jobs")
    
    # Optional if your rrServer is for example Linux and this python script should work with Windows paths
    # tcp.jobs.setPathTargetOS(1) 
    
    # we want to retrieve only job details of one user to reduce network traffic, rrServer workload and this Pythons memory.
    tcp.jobList_SetFilter4(job_user_name, "", "", rrJob._renderType.stUnset, 0, "", False, rrLib._filterIDs.isAll)

    if info_level == InfoLevel.Basic:
        # we do not need the full information about a job, this saves a bit traffic if you handle many jobs
        if not tcp.jobList_GetBasic():
            print("Error getting jobs: " + tcp.errorMessage())
            return
    elif info_level == InfoLevel.Info:
        if not tcp.jobList_GetInfo():
            print("Error getting jobs: " + tcp.errorMessage())
            return

    jobs = tcp.jobs
    print("Number of jobs in the rrServer queue: " + str(jobs.getMaxJobs()))
    jobs_count = jobs.getMaxJobsFiltered()
    print("Number of jobs - filtered by user {0}: {1}".format(job_user_name, jobs_count))
    for i in RANGE_FUNC(0, jobs_count):
        job_asMinInfo= jobs.getJobMinInfo_filterQueue(i)
        job_id = job_asMinInfo.ID
        username = job_asMinInfo.userName
        
        # This python session caches all jobs that you have in your rrServer queue.
        # But this does not mean that all data from all jobs was retrieved
        # That depends on your jobList_SetFilter() and if you have used jobList_GetBasic() or jobList_GetInfo() afterwards
        #
        # You can test the data available for a job with getJobDataType()
        job_data_type = jobs.getJobDataType(job_id)  
        # print(job_data_type)
        # possible return values:
        # jlNone    : This job ID does  not exist at the rrServer at all
        # jlMinInfo : You may use getJobMinInfo()
        # jlBasic   : You may use getJobMinInfo(), getJobBasic()
        # jlSend    : You may use getJobMinInfo(), getJobBasic(), getJobInfo()
        
        job = jobs.getJobInfo(job_id)
        # jobs.getJobInfo(job_id) returns "None" if that job does not exist with "Info" data in this session.
        # Maybe we should have checked the return of getJobDataType() before
        if (job != None):
            job_line = "\tI: {0} -\tuser: {1}\tscene: {2}\timage dim: {3}x{4}\timageDir: {5}".format(
                job.IDstr(), username, job.sceneName, job.imageWidth, job.imageHeight, job.imageDir
            )
            print(job_line)
        else:
            job = jobs.getJobBasic(job_id)
            # jobs.getJobInfo(job_id) returns "None" if that job does not exist with "Info" data in this session.
            if (job != None):
                job_line = "\tB: {0} -\tuser: {1}\tscene: {2}\timage dim: {3}x{4}\timageDir: {5}".format(
                    job.IDstr(), username, job.sceneName, job.imageWidth, job.imageHeight, job.imageDir
                )
                print(job_line)



def change_jobs_sequence(tcp, jobs, n):
    """Change in/out Sequence of n-th job in list"""
    import random
    random.seed()

    modjob_values = rrJob.getClass_SettingsOnly()
    modjob_values.seqStart=random.randint(2,50)
    modjob_values.seqEnd=random.randint(80,120)

    modjob_flags = rrJob.getClass_SettingsOnly()
    modjob_flags.seqStart=1
    modjob_flags.seqEnd=1

    modjobList = []

    job_min_info = jobs.getJobMinInfo_filterQueue(n)
    job_id = job_min_info.ID
    modjobList.append(job_id)

    print("Changing job {0} sequence to min={1}  max={2}".format(job_min_info.IDstr(), modjob_values.seqStart, modjob_values.seqEnd))
    tcp.jobModify(modjobList, modjob_values, modjob_flags)


def print_users(tcp):
    print("\nCheck users")
    if not tcp.userGet(""):  
        print("Error getting user: " + tcp.errorMessage())
        return

    users = tcp.users
    users_count = users.count()
    print("Number of user found: " + str(users_count))
    for i in RANGE_FUNC(0, users_count):
        print("\tname: " + users.at(i).name)


rrTcp = rr_connect.server_connect(user_name=None, password=None)
if not rrTcp.configGetGlobal(): #required for jobs if you want to use the OS conversion from e.g. your Linux rrServer to Windows paths
    print("ERROR getting global config: " + tcp.errorMessage())

print_clients(rrTcp)
print_jobs(rrTcp)
print_users(rrTcp)

print("\n\nConnection Stats " + rrTcp.connectionStats())
print("\n\n--- DONE ---\n\n")
