import sys, os
import rrGlobal
import  rrFT

class rrJob(object):
    '''
    this represents an intermediate class between royalrender and ftrack
    representing the interface between the two
    '''

    def __init__(self, sProjectname, sSequencename, sShotname, sPass):
        '''
        initializes an instance based on given minimal information
        '''
        self.sProjectname = sProjectname
        self.sSequencename = sSequencename
        self.sShotname = sShotname
        self.sRenderpass = sPass

        self.sSubmitname = rrSubmit.name_convention_get(sSequencename, sShotname)
        self.sName = self.name_convention_get(self.sSubmitname, sPass)
        self.sDescription = self.description_convention_get()

        self.sRRJobID = "NOT_SET"
        self.sRendercamera = "NOT_SET"
        self.sRenderstatus = "NOT_SET"
        self.sftrackID = "NOT_SET"
        self.iAverageRendertime = 0.0
        self.iAverageMemoryUsage = 0.0

        self.ftrack = None # entity in ftrack database
        self.ftrack_submit = None # entity in ftrack database
        self.ftrack_asset = None # entity in ftrack database

    @classmethod
    def from_royalrender_submitJob(cls, oRRJob):
        '''
        initializes an instance based on a royalrender job python object
        '''
        o = cls(oRRJob.companyProjectName,
                oRRJob.customSeqName,
                oRRJob.customShotName,
                oRRJob.layer)
        o.sRRJobID = oRRJob.IDstr()
        o.sRendercamera = oRRJob.camera
        o.sftrackID = oRRJob.shotgunID
        o.sRenderstatus = "None"
        return o

    @classmethod
    def from_royalrender_job(cls, oRRJob):
        '''
        initializes an instance based on a royalrender job python object
        '''
        o = cls(oRRJob.companyProjectName,
                oRRJob.customSeqName,
                oRRJob.customShotName,
                oRRJob.layer)
        o.sRRJobID = oRRJob.IDstr()
        o.sRendercamera = oRRJob.camera
        o.sftrackID = oRRJob.shotgunID

        # make status human readable
        dRenderstatus = {60: "first check", 80: "preview", 120: "rendering", 140: "postrender", 200 : "finished"}
        if dRenderstatus.has_key(oRRJob.status):
            o.sRenderstatus = dRenderstatus[oRRJob.status]
        else:
            o.sRenderstatus = oRRJob.status

        return o

    @classmethod
    def from_ftrack_id(cls, sID):
        '''
        initializes an instance based on an already existing job in ftrack (by ftrack id)
        i.e. the reversse of "from_royalrender_job()"
        '''
        #print "searching  " + str(sID) 
        ftrack_job = oFTrackSession.query(
                    'Rrjob where id is "%s"' % sID
                    ).first()

        if ftrack_job is None:
            print("could not find ftrack rrjob with ftrack ID '%s'" % sID)
            raise royalFTrackException("could not find ftrack rrjob with ftrack ID '%s'" % sID)
            return False

        o = cls(ftrack_job['project'],
                ftrack_job['parent']['parent']['parent']['name'], #sequence
                ftrack_job['parent']['parent']['name'], #shot
                ftrack_job['metadata']['rr_renderpass']) #layer

        o.sName = ftrack_job['name']
        o.sRRJobID = ftrack_job['metadata']['rr_jobid']
        o.sRendercamera = ftrack_job['metadata']['rr_rendercamera']
        o.sRenderstatus = ftrack_job['metadata']['rr_renderstatus']
        o.iAverageRendertime = ftrack_job['metadata']['rr_average_render_time']
        o.iAverageMemoryUsage = ftrack_job['metadata']['rr_average_memory_usage']
        o.sftrackID = sID

        o.ftrack = ftrack_job # entity in ftrack database
        o.ftrack_submit = ftrack_job['parent'] # entity in ftrack database

        #print "from_ftrack_id returns " + str(o) 
        return o

    @staticmethod
    def name_convention_get(sSubmitname, sJob):
        '''
        follow naming convention of shotgun integration
        '''
        return "%s_%s" % (sSubmitname, sJob)

    def description_convention_get(self):
        '''
        arbitrary convention atm (to be discussed)
        '''
        return "Royal Render job for ['%s']['%s']['%s']['%s']" % \
                (self.sProjectname, self.sSequencename, self.sShotname, self.sRenderpass)

    def ft_parent_submit_get(self):
        '''
        in ftrack this is actually quite obvious
        but adding this here as convenience
        '''
        if self.ftrack_submit is None:
            self.ftrack_submit = self.ftrack.parent
            return self.ftrack.parent
        else:
            self.ftrack_submit

    def ft_parent_shot_get(self):
        '''
        in ftrack this is actually quite obvious
        but adding this here as convenience
        '''
        return self.ftrack['parent']['parent']

    def ft_query_from_rr_jobid(self):
        '''
        ftrack query based on just the royalrender job id
        '''
        ftrack_job = oFTrackSession.query(
                    'Rrjob where metadata any ' \
                    '(key is "rr_jobid" and value is "%s")' % self.sRRJobID
                    ).first()

        if ftrack_job is None:
            print("could not find ftrack rrjob with ID '%s'" % self.sRRJobID)
            self.ftrack = None
            return False
        else:
            self.ftrack = ftrack_job
            self.sName = ftrack_job['name']
            return True

    def ft_query_from_name(self):
        '''
        ftrack query based on just the name
        '''
        ftrack_job = oFTrackSession.query(
                        'Rrjob where name is "%s"' % self.sName
                    ).first()

        if ftrack_job is None:
            print("could not find ftrack rrjob with name '%s'" % self.sName)
            self.ftrack = None
            return False
        else:
            self.ftrack = ftrack_job
            self.sName = ftrack_job['name']
            return True

    def ft_ensure(self):
        '''
        get existing ftrack rrJob based on known sRRJobID
        create a new one if it doesnt exist yet (for this the ftrack_submit
        is mandatory as a parent in the ftrack hierarchy and has to be set
        prior to calling this)
        '''

        if self.ft_query_from_name():
            print("job '%s' already existing, working with that" % self.sRRJobID)
        else:
            ftrack_job = oFTrackSession.create('Rrjob', {
                            'name': self.sName,
                            'parent': self.ftrack_submit,
                            'description': self.sDescription
                })
            oFTrackSession.commit()
            print("succesful creation of rrJob '%s'" % ftrack_job['name'])
            self.ftrack = ftrack_job


    def ft_metadata_set(self, oVersion=None):
        '''
        set metadata on the rrJob custom entity
        (or the corresponding assetversion if provided)
        '''
        if oVersion is not None:
            oEntitiy = oVersion
        else:
            oEntitiy = self.ftrack

        oEntitiy['metadata'] = {'rr_jobid' : self.sRRJobID,
                            'rr_renderpass' : self.sRenderpass,
                            'rr_rendercamera' : self.sRendercamera,
                            'rr_renderstatus': self.sRenderstatus,
                            'rr_average_render_time': self.iAverageRendertime,
                            'rr_average_memory_usage': self.iAverageMemoryUsage}

        oFTrackSession.commit()

    def ft_rr_asset_query(self):
        '''
        corresponding assets there already?
        '''
        oAsset = oFTrackSession.query(
                    'Asset where type.short is "upload" and '
                    'parent.id is "%s" and name is "%s"'
                    % (self.ftrack['id'], self.sName)
                    ).first()
        if oAsset is None:
            print("could not find ftrack rrjob asset '%s'" % self.sName)
            return False
        else:
            self.ftrack_asset = oAsset
            return True

    def ft_rr_asset_ensure(self):
        '''
        make sure we have a corresponding asset
        '''
        if not self.ft_rr_asset_query():
            print("Creating ftrack rrjob asset '%s'" % self.sName)
            oAssetType = oFTrackSession.query('AssetType where short is "upload"').first()
            oAsset = oFTrackSession.create('Asset',
                                           {'parent': self.ftrack,
                                            'name': self.sName,
                                            'type': oAssetType})
            self.ftrack_asset = oAsset

    def ft_rr_assets_iterate(self, bClear=False):
        '''
        debug/clear all assets and assetversions relating to royalrender
        from this ftrack entity
        '''

        lAssets = oFTrackSession.query('Asset where type.short is "upload" and '
                                        'parent.id is "%s"' % self.ftrack['id'])
        for oAsset in lAssets:
            print("\tAsset: %s" % oAsset['name'])

            lVersions = oFTrackSession.query(
                        'AssetVersion where asset.id is "%s"' % oAsset['id'])
            for oVersion in lVersions:
                print("\t\t Version: %s" % oVersion['comment'])
                if bClear:
                    oFTrackSession.delete(oVersion)
                    print("deleted upload assetversion '%s'" % oVersion['comment'])

            if bClear:
                oFTrackSession.delete(oAsset)
                print("deleted upload asset '%s'" % oAsset['name'])

        if bClear:
            oFTrackSession.commit()

    def ft_rr_components_create(self, lPreviewFilePaths=None, sQuicktime=None,
                                sLocation='ftrack.server'):
        '''
        creates an asset, an assetversion and components for a particular rrJob
        components will be of "main" type
        '''

        if lPreviewFilePaths is None and sQuicktime is None:
            return

        # make sure we have a corresponding asset
        self.ft_rr_asset_ensure()

        # create assetversion
        oVersion = oFTrackSession.create('AssetVersion',
                                        {'asset': self.ftrack_asset,
                                         'comment': self.sDescription})

        # also set metadata on the corresponding version
        self.ft_metadata_set(oVersion=oVersion)

        # get location now (default is to upload -- use 'ftrack.server' in that case)
        oLocation = oFTrackSession.query('Location where name is "%s"' % sLocation).one()

        # upload preview images
        if lPreviewFilePaths is not None:
            for sFilePath in lPreviewFilePaths:
                oComponent = oVersion.create_component(sFilePath,
                                                       location=oLocation)
                print("created component %s" % sFilePath)

            # set thumbnail (last of the previews taken)
            self.ftrack["thumbnail_id"] = oComponent["id"]
            oVersion["thumbnail_id"] = oComponent["id"]

        # upload quicktime
        if sQuicktime is not None:
            oComponent = oVersion.create_component(sQuicktime,
                                        {'name': self.sName},
                                        location=oLocation)
            print("created component %s" % sQuicktime)

        oFTrackSession.commit()

    def ft_notification_send(self):
        '''
        TODO: implement this (not part of the deal though)
        '''
        pass

class rrSubmit(object):
    '''
    this represents an intermediate class between royalrender and ftrack
    representing the interface between the two
    '''
    def __init__(self, sProjectname, sSequencename, sShotname):
        # mandatory information to create an ftrack entity in the specified hierarchy
        self.sProjectname = sProjectname
        self.sSequencename = sSequencename
        self.sShotname = sShotname
        self.sName = self.name_convention_get(sSequencename, sShotname)
        # these are taken from the RR job
        self.sRenderapplication = 'NOT_SET'
        self.sRenderscenename = 'NOT_SET'
        self.sUsername = 'NOT_SET'
        # these are the actual entities in the ftrack database
        # (e.g. fetched by queries)
        self.ftrack = None # submit entity in ftrack database
        self.ftrack_shot = None # parent shot entity in ftrack database

    @classmethod
    def from_royalrender_job(cls, oRRJob):
        '''
        initializes an instance based on a royalrender job python object
        '''
        o = cls(oRRJob.companyProjectName,
                oRRJob.customSeqName,
                oRRJob.customShotName)
        o.sRenderapplication = oRRJob.renderApp.name
        o.sRenderscenename = oRRJob.sceneName
        o.sUsername = oRRJob.userName

        return o

    @classmethod
    def from_ftrack_id(cls, sID):
        '''
        initializes an instance based on an already existing submit in ftrack (by ftrack id)
        i.e. the reverse of "from_royalrender_job()"
        '''
        ftrack_submit = oFTrackSession.query(
                    'Rrsubmit where id is "%s"' % sID
                    ).first()

        if ftrack_submit is None:
            raise royalFTrackException("could not find ftrack rrsubmit with ftrack ID '%s'" % sID)
            return False

        o = cls(ftrack_submit['project'],
                ftrack_submit['parent']['parent']['name'], #sequence
                ftrack_submit['parent']['name']) #shot

        o.sRenderscenename = ftrack_submit['metadata']['rr_renderscenename']
        o.sRenderapplication = ftrack_submit['metadata']['rr_render_application']
        o.sUsername = ftrack_submit['metadata']['rr_username']

        o.ftrack = ftrack_submit # entity in ftrack database
        o.ftrack_shot = ftrack_submit['parent'] # entity in ftrack database

        return o

    @staticmethod
    def name_convention_get(sSequencename, sShotname):
        '''
        follow naming convention of shotgun integration
        also add a basic timestring, so we end up with unique names for submits
        to the same shot (ftrack doesnt allow entries with the same name in the same hierarchy)
        '''
        import time
        sTime  = time.strftime("%Y-%m-%d-%H%M%S", time.localtime())

        return "%s_render_%s_%s" % (sTime, sSequencename, sShotname)

    def ft_query_self(self):
        '''
        find royalrender submission entity in ftrack
        '''
        oShot = royalFTrack.ft_shot_get(self.sProjectname,
                                     self.sSequencename,
                                     self.sShotname)
        if not oShot:
            return False
        else:
            self.ftrack_shot = oShot

        oSubmit = oFTrackSession.query(
                    'Rrsubmit where name is "%s" and '
                    'parent.id is "%s"'
                    % (self.sName, oShot['id'])
                    ).first()

        if oSubmit is None:
            print("could not find ftrack rrsubmit '%s' " \
                        "in project '%s' | sequence '%s' | shot '%s'" % \
                        (self.sName, self.sProjectname, self.sSequencename, self.sShotname)
            )
            self.ftrack = None
            return False
        else:
            self.ftrack = oSubmit
            return True

    def ft_ensure(self):
        '''
        make sure we have a royalrender submission entity in ftrack
        '''
        if self.ft_query_self():
            print("submit '%s' already existing, working with that" % self.sName)
        else:
            oSubmit = oFTrackSession.create('Rrsubmit', {
                    'name': self.sName,
                    'parent': self.ftrack_shot,
                    'description': "Royal Render submission for " \
                                "[project '%s'][sequence '%s'][shot '%s']" % \
                                (self.sProjectname, self.sSequencename, self.sShotname)
                })
            oFTrackSession.commit()
            print("succesful creation of rrSubmit '%s'" % oSubmit['name'])
            self.ftrack = oSubmit


    def ft_metadata_set(self):
        '''
        set metadata on the submission entity (we are not using custom attributes)
        '''
        self.ftrack['metadata'] = {
            'rr_render_application': self.sRenderapplication,
            'rr_renderscenename' : self.sRenderscenename,
            'rr_username' : self.sUsername
            }
        oFTrackSession.commit()

    def ft_jobs_iterate(self, bClear=False, bClearAssets=False):
        '''
        used for debugging purposes only
        '''
        for oJob in self.ftrack['children']:
            print('\t rrJob: %s [%s]' % (oJob['name'], oJob['id']))

            oRenderjob = rrJob.from_ftrack_id(oJob['id'])
            oRenderjob.ft_rr_assets_iterate(bClear=bClearAssets)

            if bClear:
                oFTrackSession.delete(oJob)
                print("deleted ftrack rrJob '%s'" % oJob['name'])

        if bClear:
            oFTrackSession.commit()


class royalFTrackException(Exception):
    '''
    This class will be used by the royalFTrack classes to define errors
    '''
    def __init__(self, msg):
        print(msg)
        rrGlobal.messageBox(rrGlobal.logLvL.warning, msg)
        super(royalFTrackException, self).__init__()


class royalFTrack(object):

    def __init__(self):
        self._imports()

    def _imports(self):
        '''
        this is importing the ftrack API module

        TODO: because of old RR py interpreter need to downgrade "requests" module
        TODO: because of old RR py interpreter need to import "encodings" before ftrack imports "json"
        TODO: get path to ftrack API from RR UI
        '''
        import encodings
        sys.path.append(rrFT.pythonPath())
        global ftrack_api
        import ftrack_api
        

    def _user_mapping_get(self, sUsername):
        '''
        represents a mapping between usernames provided by RR and usernames in ftrack
        using a dictionary where keys are RR usernames and values are ftrack usernames
        '''
        
        import json
        dUserMapping = {}
        with open("userNames.json", "r") as config_file:
            dict = json.load(config_file)        

        if dUserMapping.has_key(sUsername):
            return dUserMapping.get(sUsername)
        else:
            return sUsername

    def session_setup(self, sUsername=None):
        '''
        this sets up an appropriate ftrack session (in case of new API)
        '''
        sFTrackUser = rrFT.user()
        sFTrackURL = rrFT.url()
        sFTrackAPIkey = rrFT.apiKey()

        import requests # to except specific error (see below)
        try:
            global oFTrackSession
            oFTrackSession = ftrack_api.Session(server_url = sFTrackURL,
                                                api_key = sFTrackAPIkey,
                                                api_user = sFTrackUser,
                                                auto_connect_event_hub=True,
                                                )
        except ftrack_api.exception.ServerError as e:
            # most propably an unknown username in ftrack (or wrong API key)
            raise royalFTrackException("ftrack server error. see ftrack error message below \n\n %s" % e.message)
            return False

        except requests.exceptions.ConnectionError as e:
            raise royalFTrackException("connection error. see error message below \n\n %s" % e.message)

    @staticmethod
    def ft_shot_get(sProjectname, sSequencename, sShotname):
        '''
        get an ftrack shot based on project-, sequence- and shotname
        '''
        oProject = oFTrackSession.query('Project where name is "%s"' % sProjectname).first()
        if oProject is None:
            raise royalFTrackException("could not find ftrack project '%s'" % sProjectname)
            return False
            
        oShot=""  
        if (len(sSequencename)>0):
            oSequence = oFTrackSession.query(
                        'Sequence where name is "%s" and '
                        'project.id is "%s"'
                        % (sSequencename, oProject['id'])
                        ).first()
            if oSequence is None:
                raise royalFTrackException("could not find ftrack sequence '%s'" \
                                "in project '%s'" % (sSequencename, sProjectname))
                return False
            oShot = oFTrackSession.query(
                        'Shot where name is "%s" and '
                        'project.id is "%s" and '
                        'parent.id is "%s"'
                        % (sShotname, oProject['id'], oSequence['id'])
                        ).first()
        else:
            oShot = oFTrackSession.query(
                        'Shot where name is "%s" and '
                        'project.id is "%s"'
                        % (sShotname, oProject['id'])
                        ).first()
        
        if oShot is None:
            raise royalFTrackException("could not find ftrack shot '%s' " \
                                    "in project '%s' | sequence '%s'" % \
                                    (sShotname, sProjectname, sSequencename))
            return False
        else:
            return oShot

    def ft_submits_iterate(self, bClear=False, bClearJobs=False, bClearAssets=False):
        '''
        get an overview of what data related to royalrender is actually in ftrack
        also have the possibility to get rid of that quickly
        used for debugging and cleanup purposes only: handle with care!!
        '''
        rrSubmits = oFTrackSession.query('Rrsubmit')

        for oSubmit in rrSubmits:
            print('rrSubmit: %s' % oSubmit['name'])

            oSubmission = rrSubmit.from_ftrack_id(oSubmit['id'])
            oSubmission.ft_jobs_iterate(bClear=bClearJobs, bClearAssets=bClearAssets)

            if bClear:
                oFTrackSession.delete(oSubmit)
                print("deleted submit '%s'" % oSubmit['name'])

        if bClear:
            oFTrackSession.commit()

    def ftrack_renderstats_create(self):
        '''
        the onsubmit callback
        will create submission and corresponding jobs in ftrack
		'''
        import rr
        if rr.jobSelected_count() < 1:
            raise royalFTrackException("no job selected")
            return

        rrGlobal.progress_SetMaxA(rr.jobSelected_count() + 3)
        rrGlobal.progress_SetProgressA(0)
        rrGlobal.refreshUI()

        oJobFirst = rr.jobSelected_get(0)
        self.session_setup(oJobFirst.userName)

        rrGlobal.progress_SetProgressA(1)
        rrGlobal.refreshUI()

        # create rrSubmit / rrJobs
        oSubmission = rrSubmit.from_royalrender_job(oJobFirst)
        oSubmission.ft_ensure()
        oSubmission.ft_metadata_set()

        rrGlobal.progress_SetProgressA(2)
        rrGlobal.refreshUI()

        for iJobNr in range(0, rr.jobSelected_count()):
            oRRJob = rr.jobSelected_get(iJobNr)
            oRenderjob = rrJob.from_royalrender_submitJob(oRRJob)

            oRenderjob.ftrack_submit = oSubmission.ftrack
            oRenderjob.ft_ensure()
            oRenderjob.ft_metadata_set()
			
            #rrGlobal.writeLog(rrGlobal.logLvL.warning, "SubmitEntity is " +str(oRenderjob.ftrack["id"]),"")

            rr.jobAll_setShotgunID(iJobNr, oRenderjob.ftrack["id"])
            oRRJob.customSet_Str("FtrackID", str(oRenderjob.ftrack["id"]))
            rr.jobSelected_set(iJobNr,oRRJob)


            rrGlobal.progress_SetProgressA(iJobNr + 3)
            rrGlobal.refreshUI()

    def ftrack_renderstats_update(self):
        '''
        update metadata on rrjobs in ftrack
        '''
        import rr
        if rr.jobSelected_count() < 1:
            raise royalFTrackException("no job selected")
            return

        for iJob in range(0, rr.jobSelected_count()):

            oRRJob = rr.jobSelected_get(iJob)
            self.session_setup(oRRJob.userName)
            
            #oRenderjob = rrJob.from_royalrender_job(oRRJob)
            #if not oRenderjob.ft_query_from_rr_jobid():
            #    return
            oRenderjob = rrJob.from_ftrack_id(oRRJob.shotgunID)
            
            
            oRenderjob.iAverageRendertime = oRRJob.averageFrameTime
            oRenderjob.iAverageMemoryUsage = oRRJob.averageMemoryUsage
            dRenderstatus = {60: "first check", 80: "preview", 120: "rendering", 140: "postrender", 200 : "finished"}
            if dRenderstatus.has_key(oRRJob.status):
                oRenderjob.sRenderstatus = dRenderstatus[oRRJob.status]
            else:
                oRenderjob.sRenderstatus = oRRJob.status
            oRenderjob.ft_metadata_set()

    def ftrack_previewimages_upload(self):
        '''
        upload preview images
        '''
        import rr
        if rr.jobSelected_count() < 1:
            raise royalFTrackException("no job selected")
            return

        for iJob in range(0, rr.jobSelected_count()):

            oRRJob = rr.jobSelected_get(iJob)
            self.session_setup(oRRJob.userName)
            #oRenderjob = rrJob.from_royalrender_job(oRRJob)
            #if not oRenderjob.ft_query_from_rr_jobid():
            #    return
            #oRenderjob = rrJob.from_ftrack_id(oRRJob.shotgunID)
            oRenderjob = rrJob.from_ftrack_id(oRRJob.custom_Str("FtrackID"))
            if type(oRenderjob) is bool:
                continue

            # TODO: use oRRJob.previewNumberOfFrames to determine middle frame?
            sPicFirst = oRRJob.previewFilenameThumbnail(0)
            sPicLast = oRRJob.previewFilenameThumbnail(-1)

            if not os.path.exists(sPicFirst):
                raise royalFTrackException("preview image %s doesnt exist" % sPicFirst)
            if not os.path.exists(sPicLast):
                raise royalFTrackException("preview image %s doesnt exist" % sPicLast)

            oRenderjob.ft_rr_components_create(lPreviewFilePaths=[sPicFirst, sPicLast])

#oRRFT = royalFTrack()
#oRRFT.session_setup("testUser")
#oRRFT.ft_submits_iterate(bClear=True, bClearJobs=True, bClearAssets=True)
#oRRFT.ft_submits_iterate(bClear=False, bClearJobs=False, bClearAssets=False)