#
# Render Config Example
#
# Get and set renderorder and global config parameters using python
#
# Calling tcp.configGetGlobal() stores the Render Farm global configuration in tcp.config,
# Calling tcp.configGetRenderOrder() stores the Render Farm Priority configuration in tcp.config,
#
#


import sys
import os
import datetime

mod_dir = os.path.dirname(__file__)
if mod_dir not in sys.path:
    sys.path.append(mod_dir)

from rr_python_utils.load_rrlib import rrLib  # module libpyRR3 for python 3, libpyRR2 for python 2
import rr_python_utils.connection as rr_connect


def read_write_global_config(tcp, save_changes=False):
    print("\nRead/Write Global Config Settings\n\n")

    # Get Global Config from server
    if not tcp.configGetGlobal():
        print("Global Config not received:", tcp.errorMessage())
        return

    # read boolean setting
    print("\nAverageAbort is set to:", tcp.config.getGlobalValue("jAverageAbort"))

    # set boolean setting
    tcp.config.setGlobalValue("jAverageAbort", False)
    print("AverageAbort is now set to:", tcp.config.getGlobalValue("jAverageAbort"))

    # read int setting
    print("ProjectNameByDirStart is set to:", tcp.config.getGlobalValue("jGetProjectNameByDirStart"))

    # set int setting
    tcp.config.setGlobalValue("jGetProjectNameByDirStart", 6)
    print("ProjectNameByDirStart now is set to:", tcp.config.getGlobalValue("jGetProjectNameByDirStart"))
    
    # read float setting
    print("\nCost is set to:", tcp.config.getGlobalValue("cfgCost"))
    tcp.config.setGlobalValue("cfgCost", 1.0)
    print("Cost now is set to", tcp.config.getGlobalValue("cfgCost"))

    # string
    print("\nJobDataLocationExclude is", tcp.config.getGlobalValue("jJobDataLocationExclude"))
    tcp.config.setGlobalValue("jJobDataLocationExclude", "Foo")
    print("jJobDataLocationExclude now is", tcp.config.getGlobalValue("jJobDataLocationExclude"))

    if save_changes:
        # send configuration to the server. Server will apply the new configuration and save it to globalconfig.ini
        if tcp.configSaveGlobal():
            print("\nGlobal Configuration has been sent to be updated and saved")
        else:
            print("\nGlobal Configuration not sent", tcp.errorMessage())


def read_write_render_order(tcp, save_changes=False):
    print("\nRead/Write Render Order Config Settings\n\n")

    # Get RenderOrder Config from server
    tcp.configGetRenderOrder()

    # Render Divisions
    num_divisions = tcp.config.renderDivisions.count()
    if num_divisions == 0:
        print("\nNo Render Divisions set")
    else:
        print(f"\n {num_divisions} Render Divisions:")
        for i in range(num_divisions):
            print("\t", tcp.config.renderDivisions.at(i))
    
    new_division = "New_Division"

    if tcp.config.renderDivisions.contains(new_division):
        print(new_division, "already exists")
    else:
        print("\nAdding new division:", new_division)
        tcp.config.renderDivisions.append(new_division)
        print("Now there are", tcp.config.renderDivisions.count(), "Render Divisions")

    # Preferred Render Order
    num_preferred = tcp.config.renderPreferred.count()
    print(f"\nFound {num_preferred} Settings for preferred rendering")

    print("Adding Maya and Comp as an entry of preferred jobs")
    tcp.config.renderPreferred.append("Maya;Comp")

    print("Adding Houdini to preferred jobs")
    tcp.config.renderPreferred.append("Houdini")

    # Preferred Job types Settings (Render Order)
    print("\nPreferred Render Order")
    for i in range(tcp.config.renderPreferred.count()):
        print("\t", tcp.config.renderPreferred.at(i))
    
    # Swap two Preferred Settings: Move second preferred types to first
    print(f"\Swapping places between 0. {tcp.config.renderPreferred.at(0)} and 1. {tcp.config.renderPreferred.at(1)}")
    tcp.config.renderPreferred.swapItemsAt(0, 1)

    print("\nPreferred Render Order After Swap")
    for i in range(tcp.config.renderPreferred.count()):
        print("\t", tcp.config.renderPreferred.at(i))

    # Adding Job Privilege for client "Farm20"
    new_privilege = rrLib._rrCfgPrivilegedFor()
    new_privilege.clientName = "Farm20"
    new_privilege.list.append("blender")
    new_privilege.list.append("Comp")

    tcp.config.renderPrivileged.append(new_privilege)

    # Privileged Job Types Settings
    print("\nJob Privilege per Client Name")
    for i in range(tcp.config.renderPrivileged.count()):
        priv = tcp.config.renderPrivileged.at(i)
        print("\n\t", priv.clientName + ":")
        for job_type in priv.list.asTuple():
            print("\t\t", job_type)

        # adding Maya to Farm20 privilege
        if priv.clientName == "Farm20":
            priv.list.append("Maya")

            print(f"\n\t{priv.clientName} privileged job types now are: {priv.list.asString()}\n")

    # Exclusive Priority for Jobs; Render Exlusive has 10 slots
    print("\nExclusive priority for Project, User, RenderApp, Division:")

    # Enable/Disable Equal Priority (boolean value)
    if (tcp.config.getRenderOrderValue('EqualExclusivePrio')):
        print("\nExclusive Job Types have equal priority, changing that")
        tcp.config.setRenderOrderValue('EqualExclusivePrio', False)
    else:
        print("\nExclusive Job Types don't have equal priority, changing that")
        tcp.config.setRenderOrderValue('EqualExclusivePrio', True)
    print("Exclusive Job Types Equal Priority is now", str(tcp.config.getRenderOrderValue('EqualExclusivePrio')), "\n")

    last_empty_exclusive = None
    for i in range(tcp.config.renderExclusive.count()):
        exclusive = tcp.config.renderExclusive.at(i)

        if exclusive.list.isEmpty():
            last_empty_exclusive = i
            continue
        
        if exclusive.expire:
            expiration = "expires on " + exclusive.expire.strftime('%A %d-%m-%Y, %H:%M') 
        else:
            expiration = "never expires"

        print(f"\t{exclusive.list.asString()}, max clients: {exclusive.maxClients},", expiration)

    # Add Exclusive Priority to the last Empty slot
    if last_empty_exclusive is not None:
        exclusive = tcp.config.renderExclusive.at(last_empty_exclusive)
        exclusive.list.append("Arnold")
        exclusive.maxClients = 10

        day_after_tomorrow = datetime.datetime.now() + datetime.timedelta(days=3)
        exclusive.expire = day_after_tomorrow
    
        print(f"\nAdded new exclusive set at slot {last_empty_exclusive}:")
        print("\t", exclusive.list.asString(), "clients:", exclusive.maxClients, "expires on", exclusive.expire.strftime('%A %d-%m-%Y, %H:%M'))
    
    # Saving Config
    if save_changes:
        if tcp.configSaveRenderOrder():
            print("\nRender Order Configuration has been sent to be updated and saved")
        else:
            print("\nRender Order Configuration not sent", tcp.errorMessage())


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument("--user")

    args = parser.parse_args()

    rr_tcp = rr_connect.server_connect(user_name=args.user, askForlogin=True)
    
    read_write_global_config(rr_tcp, save_changes=False)
    read_write_render_order(rr_tcp, save_changes=False)
