# -*- coding: cp1252 -*-
#
# These lines are just information lines for the plugin loader
# rrPluginName:		KeyShot Scene parser
# rrPluginAuthor:	Paolo Acampora - Binary Alchemy
# rrPluginVersion:	%rrVersion%
# rrPluginType:		Scene Parser Plugin
# rrRRversion:		8.0
#
#
#
# This setting is important for scene parser job plugins and need to be set.
#
# rrSupportedFileExt: *.bip;
#
#
#
# Copyright (c) Holger Schoenberger - Binary Alchemy
######################################################################
import re


LUX_FILE_FORMATS = [
    ".jpg",  # lux.RENDER_OUTPUT_JPEG,   0
    ".png",  # lux.RENDER_OUTPUT_PNG,    1
    ".exr",  # lux.RENDER_OUTPUT_EXR,    2
    ".tif",  # lux.RENDER_OUTPUT_TIFF8,  3
    ".tif",  # lux.RENDER_OUTPUT_TIFF32, 4
    ".psd",  # lux.RENDER_OUTPUT_PSD8,   5
    ".psd",  # lux.RENDER_OUTPUT_PSD16,  6
    ".psd"   # lux.RENDER_OUTPUT_PSD32,  7
]


class JobData:
    scene = ''
    version = ''  # e.g. 10.1
    image_dir = ''
    image_filename = ''
    image_extension = ''

    image_padding = -1
    seq_start = -1
    seq_end = -1
    width = -1
    height = -1

    def __repr__(self):
        return str(self.__dict__)


class KeyShotParser(object):
    def __init__(self, scene_file):
        self._max_lines = 1000
        self.data = JobData()
        self.data.scene = scene_file

        self.parse_keyshot_file()

    @staticmethod
    def split_line_value(line):
        _, val = line.strip(' \t').split(' ', 1)
        return val.strip('",\n\r')

    @staticmethod
    def split_name_padding(image_filename):
        padding_re = re.compile('.\\d$')
        match = padding_re.search(image_filename)
        if match:
            padding_slice = match.start() + 1
            return image_filename[:padding_slice], int(image_filename[padding_slice:])

        return image_filename, -1

    def parse_keyshot_file(self):
        data = self.data

        with open(data.scene) as reader:
            for i, line in enumerate(reader):
                if i > self._max_lines:
                    break

                if line.startswith('//'):
                    if not data.version and 'KeyShot Scene File' in line:
                        version_re = 'v[0-9]*.[0-9]*'
                        v_info = re.findall(version_re, line)

                        if v_info:
                            data.version = v_info[0].strip('v')
                    continue

                if line.startswith('icon'):
                    continue

                line = line.strip(", ")
                if not data.image_dir and line.startswith('"render_folder"'):
                    data.image_dir = self.split_line_value(line)
                    continue

                if not data.image_filename and line.startswith('"animation_filenames"'):
                    image_filename = self.split_line_value(line)
                    data.image_filename, data.image_padding = self.split_name_padding(image_filename)
                    continue

                if not data.image_filename and line.startswith('"render_filename"'):
                    image_filename = self.split_line_value(line)
                    data.image_filename, data.image_padding = self.split_name_padding(image_filename)
                    continue

                if not data.image_dir and line.startswith('"render_job_folder"'):
                    data.image_dir = self.split_line_value(line)
                    continue

                if not data.image_filename and line.startswith('"render_job_filenames"'):
                    image_filename = self.split_line_value(line)
                    data.image_filename, data.image_padding = self.split_name_padding(image_filename)

                if data.seq_start == - 1 and line.startswith('"animation_first_frame"'):
                    data.seq_start = int(self.split_line_value(line))
                    continue

                if data.seq_end == - 1 and line.startswith('"animation_last_frame"'):
                    data.seq_end = int(self.split_line_value(line))
                    continue

                if data.width == -1 and line.startswith('"render_width"'):
                    data.width = int(self.split_line_value(line))
                    continue

                if data.height == -1 and line.startswith('"render_height"'):
                    data.height = int(self.split_line_value(line))
                    continue

                if not data.image_extension and line.startswith('"video_format"'):
                    try:
                        data.image_extension = LUX_FILE_FORMATS[int(self.split_line_value(line))]
                    except IndexError:
                        data.image_extension = '.'
                    continue


######################################################################

# Create a render app
render_app = rrJob._RenderAppBasic()
render_app.clear()

render_app.name = "KeyShot"

try:
    scene_file = rr.sceneFileToLoad()
    scene_data = KeyShotParser(scene_file).data

    render_app.setVersionBoth(scene_data.version)

    newJob = rr.getNewJob()
    newJob.sceneName = scene_file
    newJob.uiIsChecked = True
    newJob.renderApp = render_app

    if scene_data.width > 0:
        newJob.imageWidth = scene_data.width
    if scene_data.height > 0:
        newJob.imageHeight = scene_data.height

    newJob.imageDir = scene_data.image_dir
    newJob.imageFileName = scene_data.image_filename
    newJob.imageExtension = scene_data.image_extension

    newJob.splitImageFileInto_DirFileExt(False)
    newJob.seqStart = scene_data.seq_start
    newJob.seqEnd = scene_data.seq_end

    if not scene_file.endswith('.ksp'):
        newJob.customSet_Str("rrSubmitterParameter", "AllowLocalSceneCopy=0~0")

    # TODO: but requires setModelSets() and setCamera(), currently not available in headless mode
    # Render Layers in Keyshot are rendered anyway, Model Sets are more akin to separable jobs
    # newJob.layer = layerName
    # newJob.camera = cameraName

    rr.jobAll_set(999, newJob)  # use an index larger than the number of jobs to add the new one at the end of the list

except Exception as e:
    rrGlobal.writeLog2(rrGlobal.logLvL.warning, "Error parsing file (layer info): " + str(e))

    # progress messages will only be shown in the small log window of the rrSubmitter,
    # not in a dialog window: we do not want two dialog windows
    rr.returnFromPlugin(rrGlobal.pluginReturn.dataError)  # show a generic message window
    raise rrCleanExit()  # Bail out without leaving any traceback info

rr.returnFromPlugin(rrGlobal.pluginReturn.successful)
