#
# This script displays a list of jobs in a loop
# It only displays jobs that have changed since the last time
#




import sys
import os

mod_dir = os.path.dirname(__file__)
if mod_dir not in sys.path:
    sys.path.append(mod_dir)

from rr_python_utils.load_rrlib import rrLib  # module libpyRR3 for python 3, libpyRR2 for python 2
import rr_python_utils.connection as rr_connect


class JobLister(object):
    def __init__(self, user_name=None, password=None, job_user="renderfarm", job_contains="beauty"):
        """Open a tcp connection to rrServer and print out the jobs of `job_user_name`
         containing the string `job_contains`.
         Limiting the query to one user and one search string reduces network traffic, workload and memory usage

        :param user_name: username for rrServer authentication
        :param password: password  for rrServer authentication
        :param job_user_name: user who owns the job, default "render_farm"
        :param job_contains: search string to filter through jobs
        """
        self._tcp = rr_connect.server_connect(user_name, password)
        if not self._tcp.configGetGlobal(): #required for jobs if you want to use the OS conversion from e.g. your Linux rrServer to Windows paths
            print("ERROR getting global config: " + self._tcp.errorMessage())
        self._job_user_name = job_user
        self._job_contains = job_contains

        self.updateFilter()

    def updateFilter(self):
        self._tcp.jobList_SetFilter4(self._job_user_name, "", "", rrJob._renderType.stUnset, 0, self._job_contains, False, rrLib._filterIDs.isAll)

    def getNPrintJobs(self):
        """
        :param findJobID:
        :param job_user_name: user who owns the job, default "render_farm"
         Retrieve only job details of one user to reduce network traffic, workload and memory usage
        :param job_contains: Retrieve only jobs containing this string to reduce traffic, workload, usage, default "beauty"
        """
        print("\nCheck jobs")

        # Optional if your rrServer is for example Linux and this python script should work with Windows paths
        # tcp.jobs.setPathTargetOS(1) 

        last_update_time = self._tcp.lastUpdateJobs_ServerTime()
        if not self._tcp.jobList_GetInfo():
            print("Error getting jobs: " + self._tcp.errorMessage())
            return

        jobs = self._tcp.jobs
        print("Number of jobs: " + str(jobs.getMaxJobs()))
        jobs_count = jobs.getMaxJobsFiltered()
        print("Number of jobs - filtered by user '{0}' and includes '{1}': {2}".format(self._job_user_name, self._job_contains, jobs_count))

        if sys.version_info.major < 3:
            RANGE_FUNC = xrange
        else:
            RANGE_FUNC = range

        for i in RANGE_FUNC(jobs_count):
            jID = jobs.getJobMinInfo_filterQueue(i).ID
            username = jobs.getJobMinInfo_filterQueue(i).userName
            wasUpdated = jobs.getJobUpdatedTime(jID) > last_update_time
            job_type = jobs.getJobDataType(jID)

            msg = "FOUND JOB - type: {0}".format(job_type)
            if wasUpdated:
                msg += "    -UPDATED-"

            print(msg)
            if job_type in (rrJob._dataIDs.jlBasic, rrJob._dataIDs.jlSend):
                job = jobs.getJobInfo(jID)
                msg = "\tINFO - {0}\tuser: {1}\tscene: {2}\timage dim: {3}x{4}"
                msg = msg.format(job.IDstr(), username, job.sceneName, job.imageWidth, job.imageHeight)
                print(msg)


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument("-user_name")
    parser.add_argument("-password")
    parser.add_argument("-job_user")
    parser.add_argument("-job_contains")
    args = parser.parse_args()

    params = {p: getattr(args, p) for p in ("user_name", "password", "job_user", "job_contains") if getattr(args, p)}
    job_lister = JobLister(**params)

    print("Python module version: ",rrLib.getRRModuleVersion())
    print("TCP class version: ",rrLib.getTCPVersion())


    maxLoops=5

    for i in range(0, maxLoops):
        job_lister.getNPrintJobs()
