import datetime
import time
import sys
import os


mod_dir = os.path.dirname(__file__)
if mod_dir not in sys.path:
    sys.path.append(mod_dir)

from rr_python_utils.load_rrdata import rrData


SECONDS_PER_DAY = 24 * 60 * 60
MINUTES_PER_DAY = 24 * 60


def load_saved_stats():
    stats_file = rrData.getRRFolder_cfgStats() + "server_stats.db"
    print("loading stat file " + stats_file)
    rrGlobalStats = rrData._rrGlobalStatsLS()
    loading_success = rrGlobalStats.loadFromFile(stats_file)
    if not loading_success:
        print("Error loading stats" + rrGlobalStats.getError())
        return
    
    print("----------------- TIME CONVERSION EXAMPLES ---------------------------------")
    print("C++ applications work with the system time.")
    print("The system time is based on the default unix timestamp (seconds since 00:00, Jan 1, 1970, 'The Epoch').")
    print("It can either be in UTC timezone or your local timezone.")
    print("RR stat files use the local timezone as the first second is the first second of the day, no matter where you are.")
    print("(native python) datetime :                               ", datetime.datetime.now())
    print("(native python) datetime as string:                      ", datetime.datetime.now().strftime("%d.%m.%Y, %H:%M:%S"))
    print("(native python) current time from datetime :             ", datetime.datetime.now())
    print("(native python) current time from localtime() as struct: ", time.localtime())
    print("(native python) system time (UTC timezone!):             ", time.mktime(time.localtime()))
    if (sys.version_info.major == 3):
        print("(native python) datetime as system time (UTC timezone!): ", datetime.datetime.now().timestamp())

    print("(RR) currentTime_Unix_UTC() system time:                 ", rrGlobalStats.currentTime_Unix_UTC())
    print("(RR) currentTime_Unix() system time:                     ", rrGlobalStats.currentTime_Unix())
    #fromtimestamp converts a UTC system time into python datetime, it does a timezone conversion
    print("(RR+python) currentTime_Unix_UTC() to python datetime    ", datetime.datetime.fromtimestamp(rrGlobalStats.currentTime_Unix_UTC()))
    #utcfromtimestamp converts a system time into python datetime WITHOUT a timezone conversion
    print("(RR+python) currentTime_Unix()  to python datetime       ", datetime.datetime.utcfromtimestamp(rrGlobalStats.currentTime_Unix()))
    
    print("(RR) currentTime_datetime() exported as python datetime: ", rrGlobalStats.currentTime_datetime())

    print("-----------------------------------------------------------------------------")
    return rrGlobalStats


def print_24h_stats(rrGlobalStats):
    print("   ")
    print(" -------------- 24h stats ------------------  ")
    print("24h stats are saved every minute a day. Slot 0 is always the time 00:00-00:01 no matter the current time.")
    print("Last saved stat, localtime ",          rrGlobalStats.lastSaveDay_asDatetime())
    print("Last saved stat, localtime, seconds since 'the epoch':  ", rrGlobalStats.lastSaveDay_asUnixTime())
    print("Last saved stat, localtime, seconds since midnight:     ", rrGlobalStats.lastSaveDay_asUnixTime() % SECONDS_PER_DAY)
    print("Last saved stat, localtime, minutes since midnight:     ", int((rrGlobalStats.lastSaveDay_asUnixTime() % SECONDS_PER_DAY) / 60) )
    print("Last saved stat, localtime, stat slots since midnight:  ", int((rrGlobalStats.lastSaveDay_asUnixTime() % SECONDS_PER_DAY) / rrGlobalStats.secDiff_Day()) )

    last_day_slot = rrGlobalStats.lastSaveDay_asUnixTime() % SECONDS_PER_DAY
    if sys.version_info.major < 3:
        last_day_slot = last_day_slot / rrGlobalStats.secDiff_Day()
    else:
        # last_day_slot needs to be integer
        last_day_slot = last_day_slot // rrGlobalStats.secDiff_Day()


    total_frames = 0
    for i in range(0, rrGlobalStats.dayMax()):
        total_frames += rrGlobalStats.day(i).framesPerMin
    print("Frames rendered in last 24 hours: ", total_frames)
    print("Frames left (almost) now:       ", rrGlobalStats.day(last_day_slot).framesLeft)

    print("   ")


def print_30day_stats(rr_save_stats):
    print(" -------------- 30day stats ------------------ ")
    print("30day stats are saved every 30 minutes. The last slot in the array contains the last 'completed' 30 min (if rrServer is running).")
    print("Last saved stat, localtime ",          rrGlobalStats.lastSaveMonth_asDatetime())
    print("Last saved stat, localtime, days since 'the epoch':     ", rrGlobalStats.lastSaveMonth_asUnixTime() / SECONDS_PER_DAY)
    print("Last saved stat, slots since 'the epoch'. (localtime /30/60)",   rrGlobalStats.lastSaveMonth)

    #last day saved:
    slot_asday = int(rrGlobalStats.lastSaveMonth_asUnixTime() / SECONDS_PER_DAY)
    #last saved. Not as second or day, as 'slots' since the epoch:
    slot_asTotalSlots = rrGlobalStats.lastSaveMonth 
    

    total_frames_all = 0
    total_frames3_d = 0
    total_frames_comp = 0
    total_frames_slot_count = 0

    for monthSlot in range(rr_save_stats.monthMax() - 1, 0, -1): #loop from current time into the past
        new_slot_asday = int(slot_asTotalSlots * rrGlobalStats.secDiff_Month() / SECONDS_PER_DAY)
        if new_slot_asday != slot_asday: #we reached a new day, so print day collected
            total_frames_rendered_day_all = 0
            total_frames_rendered_day3_d = 0
            total_frames_rendered_day_comp = 0
            if total_frames_slot_count > 0:
                # framesPer90s is the average number of frames rendered in this time segment.
                # First we calculate the average of all stats taken this day:
                total_frames_all = float(total_frames_all) / total_frames_slot_count
                total_frames3_d = float(total_frames3_d) / total_frames_slot_count
                total_frames_comp = float(total_frames_comp) / total_frames_slot_count
                # now we multiple it by the ammount of 90s slots each day to convert framesPer90s into "frames per day"
                minutes_per_day = SECONDS_PER_DAY / 60
                total_frames_rendered_day_all = total_frames_all * minutes_per_day
                total_frames_rendered_day3_d = total_frames3_d * minutes_per_day
                total_frames_rendered_day_comp = total_frames_comp * minutes_per_day

            somePrint=False
            if (total_frames_all>0):
                somePrint=True
                print(datetime.datetime.utcfromtimestamp(slot_asday * SECONDS_PER_DAY).date(),
                  "   average number of      frames rendered per minute: %5.2f" % total_frames_all)
            if (total_frames3_d>0):
                somePrint=True  
                print(datetime.datetime.utcfromtimestamp(slot_asday * SECONDS_PER_DAY).date(),
                  "   average number of 3D   frames rendered per minute: %5.2f" % total_frames3_d)
            if (total_frames_comp>0):
                somePrint=True
                print(datetime.datetime.utcfromtimestamp(slot_asday * SECONDS_PER_DAY).date(),
                  "   average number of Comp frames rendered per minute: %5.2f" % total_frames_comp)
            
            if (total_frames_rendered_day_all>0):
                somePrint=True
                print(datetime.datetime.utcfromtimestamp(slot_asday * SECONDS_PER_DAY).date(),
                  "   ~ number of      frames rendered this day: %6.1f" % total_frames_rendered_day_all)
            if (total_frames_rendered_day3_d>0):
                somePrint=True
                print(datetime.datetime.utcfromtimestamp(slot_asday * SECONDS_PER_DAY).date(),
                  "   ~ number of 3D   frames rendered this day: %6.1f" % total_frames_rendered_day3_d)
            if (total_frames_rendered_day_comp>0):
                somePrint=True
                print(datetime.datetime.utcfromtimestamp(slot_asday * SECONDS_PER_DAY).date(),
                  "   ~ number of Comp frames rendered this day: %6.1f" % total_frames_rendered_day_comp)
            
            if somePrint:
                print("                  ")
            else: 
                print("  -                          -                         -")

            total_frames3_d = 0
            total_frames_all = 0
            total_frames_comp = 0
            slot_asday = new_slot_asday

        #sum information of this day
        total_frames_all += rr_save_stats.month(monthSlot).framesPerMin
        total_frames3_d += rr_save_stats.month(monthSlot).framesPerMin_3D
        total_frames_comp += rr_save_stats.month(monthSlot).framesPerMin_Comp
        total_frames_slot_count += 1
        slot_asTotalSlots -= 1 #instead we could calculate (rrGlobalStats.lastSaveMonth + rr_save_stats.monthMax() - 2 - monthSlot) every time as well


def print_year_stats(rr_save_stats):
    print("   ")
    print(" -------------- years stats ------------------ ")
    slot_time_diff_seconds = 6 * 60 * 60
    print("Last saved stat, localtime, converted to UTC datetime     ",
          datetime.datetime.utcfromtimestamp(rr_save_stats.lastSaveYear * slot_time_diff_seconds))
    print("Last saved stat, localtime, seconds since 'the epoch':    ", rr_save_stats.lastSaveYear * slot_time_diff_seconds)
    print("Last saved stat, localtime, days since 'the epoch':       ",
          rr_save_stats.lastSaveYear * slot_time_diff_seconds / SECONDS_PER_DAY)

    year_slot_time = rr_save_stats.lastSaveYear
    year_slot_day = year_slot_time * slot_time_diff_seconds / SECONDS_PER_DAY


rrGlobalStats = load_saved_stats()

if rrGlobalStats:
    print_24h_stats(rrGlobalStats)
    print_30day_stats(rrGlobalStats)
    print_year_stats(rrGlobalStats)
